package ga.view.aerial;

import ga.core.GA;
import ga.core.algorithm.interactive.ISIGA;
import ga.core.evaluation.EvaluationListener;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.individual.IDebugInfo;
import ga.core.individual.IIndividual;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;
import ga.view.factory.EffectsFactory;
import ga.view.input.CamDragListener;
import ga.view.interfaces.IPhenotypeGenerator;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;
import ga.view.streaming.showroom.ShowRoomFactory;
import ga.view.streaming.showroom.ShowRoomSettings;
import ga.view.streaming.showroom.ShowRoomSettings.ShowRoomType;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.math.Vector3f;
import com.jme3.scene.Spatial;
import com.jme3.system.AppSettings;

/**
 * This interactive evaluator provides an aerial view on the show room. There
 * are also 5 buttons for assigning fitness.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class AerialEvaluationState<T extends IIndividual<T>> extends SceneState
    implements IInteractiveFitnessEvaluator<T>, IMenuListenerParent {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(AerialEvaluationState.class.getName());

  private final IPhenotypeGenerator<T, ? extends Spatial> phenotypeGenerator;
  private ShowRoomFactory showRoomFactory;

  private AerialUIState<T> uiState;
  // private ShowRoomState<T> showRoomState;

  private final List<EvaluationListener<T>> listeners = new ArrayList<EvaluationListener<T>>();

  private CameraSettings camSettings;

  private MenuListener menuListener;

  private AppSettings settings;

  private CamDragListener dragListener;

  private ISIGA<T> algorithm;

  private T individual;

  private Application app;

  /**
   * Creates a new aerial evaluation state. This must not be called directly,
   * the configuration parser will instantiate this class.
   * 
   * @param phenotypeGenerator
   *          The phenotype generator.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public AerialEvaluationState(
      final IPhenotypeGenerator<T, ? extends Spatial> phenotypeGenerator) {
    this.phenotypeGenerator = phenotypeGenerator;
  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    this.app = app;
    phenotypeGenerator.setAssetManager(assetManager);

    this.settings = app.getContext().getSettings();

    ShowRoomSettings srSettings;

    final Object o = algorithm.getContext().get(GA.KEY_VALIDATION_SPACE);
    if (o != null && o instanceof ShowRoom) {
      srSettings = ((ShowRoom) o).getShowRoomSettings();
    } else {
      srSettings = new ShowRoomSettings();

      srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
      srSettings.put(ShowRoomSettings.BOX_WIDTH, 6f);
      srSettings.put(ShowRoomSettings.BOX_LENGTH, 4f);
      srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);
    }

    showRoomFactory = new ShowRoomFactory(assetManager, settings, srSettings);
    final ShowRoom showRoom = showRoomFactory.createShowRoom();

    camSettings = CameraSettings.getOrthographicSettings(showRoom, settings);

    // showRoomState = new ShowRoomState<T>(this, camSettings);

    initCam();

    algorithm.getContext().put(GA.KEY_VALIDATION_SPACE, showRoom);

    // TODO these are ga settings and should be put to the config file
    algorithm.getContext().put(GA.KEY_GENOME_MIN_LENGTH, 2);
    algorithm.getContext().put(GA.KEY_GENOME_MAX_LENGTH, 10);
    algorithm.getContext().put(GA.KEY_EVALUATED_INDIVIDUALS_MIN_COUNT, 10);
    algorithm.getContext().put(GA.KEY_INTERVAL_FITNESS_MAX_WIDTH, 100f);
    algorithm.init();

    dragListener = new CamDragListener(cam, inputManager, camSettings);

    uiState = new AerialUIState<T>(this, menuListener);
    stateManager.attach(uiState);

    inputManager.addMapping("switch view", new KeyTrigger(KeyInput.KEY_V));
    inputManager.addListener(new ViewSwitchListener(), "switch view");

    EffectsFactory.addShadowProcessor(assetManager, settings, viewPort,
        new Vector3f(3.0f, 2.1f, 3.0f));
    EffectsFactory.addLightScatteringProcessor(assetManager, inputManager,
        settings, viewPort, new Vector3f(3.0f, 1.9f, 3.0f));
    EffectsFactory.addSSAOProcessor(assetManager, inputManager, settings,
        viewPort);
  }

  @Override
  public void cleanup() {
    super.cleanup();

    if (algorithm != null) {
      algorithm.exit();
    }

    // if (showRoomState != null) {
    // showRoomState.setEnabled(false);
    // stateManager.detach(showRoomState);
    // }

    stateManager.detach(uiState);
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    if (dragListener != null) {
      dragListener.setEnabled(enabled);
    }

    if (uiState != null) {
      uiState.setEnabled(enabled);
    }
  }

  /**
   * Configures the camera.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initCam() {
    camSettings.configureCamera(cam);
  }

  @Override
  public void setAlgorithm(final ISIGA<T> algorithm) {
    this.algorithm = algorithm;
  }

  @Override
  public ISIGA<T> getAlgorithm() {
    return algorithm;
  }

  @Override
  public void evaluate(final T individual) {
    if (individual instanceof IDebugInfo) {
      LOGGER.log(Level.WARNING, ((IDebugInfo) individual).getGenotypeString());
    } else {
      LOGGER.log(Level.WARNING, individual.toString());
    }

    this.individual = individual;
    rootNode.detachAllChildren();
    final Spatial phenotype = phenotypeGenerator.createPhenotype(individual);

    final ShowRoom showRoom = showRoomFactory.createShowRoom();
    showRoom.setPhenotype(phenotype);
    rootNode.attachChild(showRoom);

    uiState.setIndividual(individual);
  }

  @Override
  public void fireNewIndividualRequested() {
    for (final EvaluationListener<T> l : listeners) {
      l.newIndividualRequested();
    }
  }

  @Override
  public void fireIndividualEvaluated(final T individual) {
    for (final EvaluationListener<T> l : listeners) {
      l.individualEvaluated(individual);
    }
  }

  @Override
  public void addEvaluationListener(final EvaluationListener<T> listener) {
    listeners.add(listener);
  }

  @Override
  public void removeEvaluationListener(final EvaluationListener<T> listener) {
    listeners.remove(listener);
  }

  /**
   * Gets the drag listener.
   * 
   * @return the drag listener
   */
  public CamDragListener getDragListener() {
    return dragListener;
  }

  /**
   * Keyboard listener to switch the perspective.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ViewSwitchListener implements ActionListener {
    private int i = 1;

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        final int length = CameraSettings.Type.values().length;

        final CameraSettings.Type type = CameraSettings.Type.values()[++i
            % length];

        camSettings = CameraSettings.getSettings(type,
            showRoomFactory.createShowRoom(), settings);

        camSettings.configureCamera(cam);
        dragListener.init(camSettings);

        LOGGER.info("Camera is now " + type);
      }
    }
  }
}
